/**  @file bta_filter_motion_detector.c
*  
*    @brief This file implements the motion detector filter
*  
*    BLT_DISCLAIMER
*  
*    @author Alex Falkensteiner
*  
*    @cond svn
*  
*    Information of last commit
*    $Rev::               $:  Revision of last commit
*    $Author::            $:  Author of last commit
*    $Date::              $:  Date of last commit
*  
*    @endcond
*/


#include "bta_motion_detector.h"
#include <stdlib.h>
#include <string.h>

#ifndef BTA_EXCLUDE_FILTERS

// Local prototypes
static BTA_Status reset(BTA_FltMotionDetectorInst *inst, BTA_InfoEventInst *infoEventInst);


BTA_Status BFLTmotionDetectorInit(BTA_FltMotionDetectorConfig *config, BTA_FltHandle *handle, BTA_InfoEventInst *infoEventInst) {
    BTA_Status status;
    BTA_FltMotionDetectorInst *inst;
    if (!config || !handle) {
        return BTA_StatusInvalidParameter;
    }
    *handle = 0;
    inst = (BTA_FltMotionDetectorInst *)malloc(sizeof(BTA_FltMotionDetectorInst));
    if (!inst) {
        return BTA_StatusOutOfMemory;
    }
    inst->infoEventInst = infoEventInst;
    inst->slafWindowLength = config->slafWindowLength;
    inst->slafStride = config->slafStride;
    inst->slafStrideCounter = 0;
    inst->threshold = config->threshold;
    inst->sum = 0;
    inst->avg = 0;
    status = BVQinit(inst->slafWindowLength, BTA_QueueModeDropOldest, &(inst->channelQueueInst));
    if (status != BTA_StatusOk) {
        free(inst);
        return status;
    }
    *handle = inst;
    return BTA_StatusOk;
}


BTA_Status BFLTmotionDetectorClose(BTA_FltHandle *handle) {
    BTA_FltMotionDetectorInst **inst = (BTA_FltMotionDetectorInst **)handle;
    if (!inst) {
        return BTA_StatusInvalidParameter;
    }
    if (!*inst) {
        return BTA_StatusOk;
    }
    BVQclose(&((*inst)->channelQueueInst), &BTAfreeChannelPrivate);
    BTAfreeChannel(&((*inst)->avg));
    BTAfreeChannel(&((*inst)->sum));
    free(*inst);
    *inst = 0;
    return BTA_StatusOk;
}


BTA_Status BFLTmotionDetectorApply(BTA_FltHandle handle, BTA_Frame **frame) {
    BTA_FltMotionDetectorInst *inst = (BTA_FltMotionDetectorInst *)handle;
    uint8_t chIn;
    uint32_t count;
    BTA_Status status;
    BTA_Channel *channelTemp = 0;
    BTA_Channel *channelInput = 0;
    BTA_Channel *channelOut = 0;
    if (!inst || !frame) {
        return BTA_StatusInvalidParameter;
    }
    if (!*frame) {
        return BTA_StatusInvalidParameter;
    }
    for (chIn = 0; chIn < (*frame)->channelsLen; chIn++) {
        if ((*frame)->channels[chIn]->id == BTA_ChannelIdDistance) {
            BTAcloneChannel((*frame)->channels[chIn], &channelInput);
            break;
        }
    }
    if (!channelInput) {
        BTAinfoEventHelper(inst->infoEventInst, 5, BTA_StatusInvalidParameter, "BFLTmotionDetectorApply: No distance channel found", 0);
        return BTA_StatusInvalidParameter;
    }
    if (!inst->sum) {
        status = BTAcloneChannel(channelInput, &inst->sum);
        if (status != BTA_StatusOk) {
            BTAfreeChannel(&channelInput);
            return status;
        }
        // new data format requires new malloc for data
        free(inst->sum->data);
        inst->sum->dataFormat = BTA_DataFormatSInt32;
        inst->sum->dataLen = inst->sum->xRes * inst->sum->yRes * sizeof(uint32_t);
        inst->sum->data = (uint8_t *)calloc(inst->sum->dataLen, sizeof(uint8_t));
        if (!inst->sum->data) {
            BTAfreeChannel(&channelInput);
            return status;
        }
    }

    status = BVQgetCount(inst->channelQueueInst, &count);
    if (status != BTA_StatusOk) {
        BTAfreeChannel(&channelInput);
        return status;
    }

    if (count > 0) {    
        status = BTAsubtChannel(inst->avg, channelInput, &channelOut, inst->infoEventInst);
        if (status != BTA_StatusOk) {
            BTAfreeChannel(&channelInput);
            return status;
        }
        status = BTAthresholdInPlace(channelOut, inst->threshold, 1, inst->infoEventInst);
        if (status != BTA_StatusOk) {
            BTAfreeChannel(&channelOut);
            BTAfreeChannel(&channelInput);
            return status;
        }
        BTAchangeDataFormat(channelOut, BTA_DataFormatUInt16, inst->infoEventInst);
        channelOut->id = BTA_ChannelIdFlags;
        status = BTAinsertChannelIntoFrame(*frame, channelOut);
        if (status != BTA_StatusOk) {
            BTAfreeChannel(&channelOut);
            BTAfreeChannel(&channelInput);
            return status;
        }
    }
    if (inst->avg) {
        status = BTAcloneChannel(inst->avg, &channelOut);
        if (status != BTA_StatusOk) {
            BTAfreeChannel(&channelInput);
            return status;
        }
        status = BTAinsertChannelIntoFrame(*frame, channelOut);
        if (status != BTA_StatusOk) {
            BTAfreeChannel(&channelOut);
            BTAfreeChannel(&channelInput);
            return status;
        }
    }

    if (count == inst->slafWindowLength && inst->slafStrideCounter >= inst->slafStride) {
        inst->slafStrideCounter = 0;
        status = BVQdequeue(inst->channelQueueInst, (void **)&channelTemp, 1);
        if (status != BTA_StatusOk) {
            BTAinfoEventHelper(inst->infoEventInst, 5, status, "BFLTmotionDetectorApply: Error in dequeue", 0);
            BTAfreeChannel(&channelInput);
            return status;
        }
        count--;

        status = BTAsubtChannelInPlace(inst->sum, channelTemp, inst->infoEventInst);
        BTAfreeChannel(&channelTemp);
        if (status != BTA_StatusOk) {
            BTAfreeChannel(&channelInput);
            BTAfreeFrame(frame);
            reset(inst, inst->infoEventInst);
            return status;
        }
    }

    if (count < inst->slafWindowLength) {
        status = BTAaddChannelInPlace(inst->sum, channelInput, inst->infoEventInst);
        if (status != BTA_StatusOk) {
            BTAfreeChannel(&channelInput);
            return status;
        }
        status = BVQenqueue(inst->channelQueueInst, channelInput, &BTAfreeChannelPrivate);
        if (status != BTA_StatusOk) {
            BTAfreeChannel(&channelInput);
            return status;
        }
        count++;
        BTAfreeChannel(&inst->avg);
        status = BTAdivideChannelByNumber(inst->sum, count, &inst->avg, inst->infoEventInst);
        if (status != BTA_StatusOk) {
            return status;
        }
    }
    else {
        BTAfreeChannel(&channelInput);
    }
    inst->slafStrideCounter++;
    return BTA_StatusOk;
}


static BTA_Status reset(BTA_FltMotionDetectorInst *inst, BTA_InfoEventInst *infoEventInst) {
    BTA_Status status;
    BTAfreeChannel(&(inst->sum));
    status = BVQclear(inst->channelQueueInst, &BTAfreeChannelPrivate);
    return status;
}



#endif